<?php
// app.php
// Single-file app: serves UI and API endpoints (add/list/delete).
// Data stored in data/customers.json (file-based, no SQL).

// --- CONFIG ---
$data_file = __DIR__ . '/data/customers.json';
// Ensure data directory exists
$data_dir = dirname($data_file);
if (!is_dir($data_dir)) {
    @mkdir($data_dir, 0755, true);
    // create empty json file if not present
    if (!file_exists($data_file)) file_put_contents($data_file, json_encode([]));
}

// Helper: read JSON
function read_data($file) {
    if (!file_exists($file)) return [];
    $json = @file_get_contents($file);
    $arr = json_decode($json, true);
    return is_array($arr) ? $arr : [];
}

// Helper: write JSON (atomic)
function write_data($file, $arr) {
    $tmp = $file . '.tmp';
    file_put_contents($tmp, json_encode($arr, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE), LOCK_EX);
    rename($tmp, $file);
}

// Handle API calls (AJAX)
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

if ($method === 'POST' && $action === 'add') {
    // Expect fields: customer_name, subscription_time, due_date (datetime-local), note
    $name = trim($_POST['customer_name'] ?? '');
    $subscription_time = trim($_POST['subscription_time'] ?? '');
    $due_date_raw = trim($_POST['due_date'] ?? '');
    $note = trim($_POST['note'] ?? '');

    if ($name === '' || $due_date_raw === '') {
        http_response_code(400);
        echo json_encode(['status'=>'error','message'=>'Customer name and due date required.']);
        exit;
    }

    // Parse datetime-local from browser (format: YYYY-MM-DDTHH:MM or with seconds)
    // Treat it as Asia/Kolkata timezone (per your preference).
    $dt = DateTime::createFromFormat('Y-m-d\TH:i', $due_date_raw, new DateTimeZone('Asia/Kolkata'));
    if (!$dt) {
        // try with seconds
        $dt = DateTime::createFromFormat('Y-m-d\TH:i:s', $due_date_raw, new DateTimeZone('Asia/Kolkata'));
    }
    if (!$dt) {
        http_response_code(400);
        echo json_encode(['status'=>'error','message'=>'Invalid due date format.']);
        exit;
    }
    $due_mysql = $dt->format('Y-m-d H:i:s');

    // Build record
    $record = [
        'id' => uniqid('', true),
        'customer_name' => $name,
        'subscription_time' => $subscription_time,
        'due_date' => $due_mysql,      // stored as IST string
        'note' => $note,
        'notified' => false,
        'created_at' => (new DateTime('now', new DateTimeZone('Asia/Kolkata')))->format('Y-m-d H:i:s')
    ];

    $data = read_data($data_file);
    array_unshift($data, $record); // newest first
    write_data($data_file, $data);

    echo json_encode(['status'=>'ok','message'=>'Customer added','record'=>$record]);
    exit;
}

if ($method === 'GET' && $action === 'list') {
    $data = read_data($data_file);
    echo json_encode(['status'=>'ok','data'=>$data]);
    exit;
}

if ($method === 'POST' && $action === 'delete') {
    // Expect id
    $id = $_POST['id'] ?? '';
    if ($id === '') {
        http_response_code(400);
        echo json_encode(['status'=>'error','message'=>'ID required']);
        exit;
    }
    $data = read_data($data_file);
    $new = array_values(array_filter($data, function($r) use ($id){ return ($r['id'] ?? '') !== $id; }));
    write_data($data_file, $new);
    echo json_encode(['status'=>'ok','message'=>'Deleted']);
    exit;
}

// If not API, show UI (HTML)
?><!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Customers — Subscriptions</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body{padding:20px; background:#f8f9fa;}
    .card{box-shadow:0 6px 18px rgba(0,0,0,0.06);}
    .note-small{white-space:pre-wrap; font-size:0.95rem;}
    @media (max-width:576px){
      .btn-row{flex-direction:column;}
      .btn-row .btn{width:100%; margin-bottom:8px;}
    }
  </style>
</head>
<body>
  <div class="container">
    <div class="d-flex align-items-center justify-content-between mb-3">
      <h3 class="m-0">Subscription Manager</h3>
      <small class="text-muted">Responsive • File-based JSON • Telegram reminders</small>
    </div>

    <div class="row g-3">
      <div class="col-lg-5">
        <div class="card p-3">
          <h5>Add Customer</h5>
          <form id="addForm">
            <div class="mb-2">
              <label class="form-label">Customer Name *</label>
              <input type="text" name="customer_name" class="form-control" required>
            </div>
            <div class="mb-2">
              <label class="form-label">Subscription Time (optional)</label>
              <input type="text" name="subscription_time" class="form-control" placeholder="e.g., 1 month">
            </div>
            <div class="mb-2">
              <label class="form-label">Due Date & Time (IST) *</label>
              <input type="datetime-local" name="due_date" class="form-control" required>
            </div>
            <div class="mb-2">
              <label class="form-label">Note 📝</label>
              <textarea name="note" rows="3" class="form-control"></textarea>
            </div>
            <div class="d-flex btn-row">
              <button class="btn btn-primary me-2" type="submit">Add Customer</button>
              <button type="button" class="btn btn-outline-secondary" id="refreshBtn">Refresh List</button>
            </div>
            <div id="msg" class="mt-2"></div>
          </form>
        </div>
      </div>

      <div class="col-lg-7">
        <div class="card p-3">
          <div class="d-flex justify-content-between align-items-center mb-2">
            <h5 class="m-0">Customers</h5>
            <small class="text-muted">Showing latest first</small>
          </div>

          <div id="listArea" class="table-responsive" style="max-height:60vh; overflow:auto;">
            <!-- Table injected by JS -->
          </div>

          <div class="mt-3 text-muted">
            <strong>Note:</strong> To enable automatic Telegram notifications, add the provided <code>notify_due.php</code> as a cron job on your server (instructions below).
          </div>
        </div>
      </div>
    </div>

    <footer class="mt-3 text-center text-muted">
      Created for your panel — file storage at <code>data/customers.json</code>
    </footer>
  </div>

<script>
async function api(action, method='GET', body=null) {
  let url = '?action=' + action;
  const opts = {method: method};
  if (body && !(body instanceof FormData)) {
    opts.headers = {'Content-Type':'application/x-www-form-urlencoded; charset=UTF-8'};
    opts.body = new URLSearchParams(body).toString();
  } else if (body instanceof FormData) {
    opts.body = body;
  }
  const res = await fetch(url, opts);
  return res.json();
}

function renderTable(items) {
  if (!items || items.length===0) {
    document.getElementById('listArea').innerHTML = '<div class="p-3 text-center text-muted">No customers yet.</div>';
    return;
  }
  let html = '<table class="table table-sm table-hover align-middle"><thead><tr><th>Name</th><th>Subscription</th><th>Due (IST)</th><th>Note</th><th>Notified</th><th></th></tr></thead><tbody>';
  items.forEach(item => {
    html += `<tr>
      <td>${escapeHtml(item.customer_name)}</td>
      <td>${escapeHtml(item.subscription_time||'')}</td>
      <td>${escapeHtml(item.due_date)}</td>
      <td class="note-small">${escapeHtml(item.note||'')}</td>
      <td>${item.notified ? '<span class="badge bg-success">Yes</span>' : '<span class="badge bg-warning text-dark">No</span>'}</td>
      <td><button class="btn btn-sm btn-danger" onclick="del('${item.id}')">Delete</button></td>
    </tr>`;
  });
  html += '</tbody></table>';
  document.getElementById('listArea').innerHTML = html;
}

function escapeHtml(s) {
  if (!s) return '';
  return s.replaceAll('&','&amp;').replaceAll('<','&lt;').replaceAll('>','&gt;').replaceAll('"','&quot;');
}

async function loadList(){
  const r = await api('list','GET');
  if (r.status === 'ok') {
    renderTable(r.data);
  } else {
    document.getElementById('listArea').innerHTML = '<div class="p-3 text-danger">Failed to load.</div>';
  }
}

document.getElementById('addForm').addEventListener('submit', async function(e){
  e.preventDefault();
  const form = e.target;
  const data = new FormData(form);
  document.getElementById('msg').innerHTML = 'Saving...';
  const r = await fetch('?action=add', {method:'POST', body: data});
  const j = await r.json();
  if (j.status === 'ok') {
    document.getElementById('msg').innerHTML = '<div class="text-success">Added successfully.</div>';
    form.reset();
    loadList();
  } else {
    document.getElementById('msg').innerHTML = '<div class="text-danger">'+ (j.message || 'Error') +'</div>';
  }
});

document.getElementById('refreshBtn').addEventListener('click', loadList);

async function del(id){
  if (!confirm('Delete this customer?')) return;
  const body = new FormData();
  body.append('id', id);
  const r = await fetch('?action=delete', {method:'POST', body: body});
  const j = await r.json();
  if (j.status === 'ok') {
    loadList();
  } else {
    alert('Delete failed');
  }
}

// initial load
loadList();
</script>

</body>
</html>